/**
 * As agreed https://cordial.atlassian.net/browse/SF-92
 *
 * One-time customers export job:
 * Exports customers that are modified after cordialCustomersDateUpdate.
 * Does not update lastExportDate of customers.
 *
 * Recurring customers export job:
 * Exports customers that are modified after cordialCustomersDateUpdate
 *      AND lastExportDate is before last modified date.
 * Updates lastExportDate of each customer.
 **/

'use strict';

var CustomerMgr = require('dw/customer/CustomerMgr');
var Site = require('dw/system/Site');
var Transaction = require('dw/system/Transaction');
var Logger = require('dw/system/Logger');
var CustomObjectMgr = require('dw/object/CustomObjectMgr');
var CustomerMapping = require('~/cartridge/scripts/lib/mapping/customer');
var Calendar = require('dw/util/Calendar');
var helper = require('~/cartridge/scripts/lib/mapping/helper/helper');
var FileWriter = require('dw/io/FileWriter');
var File = require('dw/io/File');
var CSVStreamWriter = require('dw/io/CSVStreamWriter');
var cordial = require('~/cartridge/scripts/services/cordial');
var errorFlag = false;

function callCordialUpdateContacts(contact) {
      var errorFlag = false;
      cordial.UpdateContacts(contact, function () {
          Transaction.wrap(function () {
            contact.custom.cordialLastExported = helper.getTime();
          });
      }, function (response) {
          errorFlag = true;
          Logger.error(['Import customer error on update', response]);
          Logger.error(['exportCustomers--contact-data', contact.email,
            contact.custom.cordialCID, contact.custom.cordialLastExported,
            contact.lastModified]);
      });
      return errorFlag;
}

/**
 * This function export customers
 **/
function exportCustomers(options) {
      var searchProfiles;
      var counter = 0;
      var limit = 1000;

      var lastRunObject = CustomObjectMgr.getCustomObject('CORDIAL_LASTRUN', 'cordialCustomersDateUpdate');
      if (!lastRunObject) {
        Transaction.wrap(function () {
            lastRunObject =
                CustomObjectMgr.createCustomObject('CORDIAL_LASTRUN', 'cordialCustomersDateUpdate');
        });
      }

      var cordialCustomersDateUpdate = lastRunObject.custom.lastRunTime
            ? lastRunObject.custom.lastRunTime
            : Site.getCurrent().getCustomPreferenceValue('cordialCustomersDateUpdate');

      var newCordialCustomersDateUpdate = helper.getTime();

      var errorFlag = false;
      var direction = 'none';

      if (cordialCustomersDateUpdate) {
            searchProfiles = CustomerMgr.searchProfiles(
                'lastModified > {0}',
                null,
                cordialCustomersDateUpdate
            );
      } else {
            searchProfiles = CustomerMgr.searchProfiles('creationDate < {0}', null,
                newCordialCustomersDateUpdate);
      }

      Logger.info(['start-exportCustomers', cordialCustomersDateUpdate,
        searchProfiles.getCount(), direction]);

      while (searchProfiles.hasNext() && (counter < limit)) {
        var contact = searchProfiles.next();
        var doRun = false;

        if (!contact.custom.cordialLastExported) doRun = true;
        if (!contact.lastModified) doRun = true;
        if (contact.lastModified > contact.custom.cordialLastExported) doRun = true;

        if (doRun) {
            if (String(contact.custom.cordialCID).length > 5) {
                  if (callCordialUpdateContacts(contact)) {
                    errorFlag = true;
                  }
            } else {
                  Logger.info(['search--for-Contacts', contact.getCredentials().getLogin()]);

                  cordial.GetContacts({
                    'email': contact.getCredentials().getLogin()
                  }, function (response) {
                    var cordialContact = JSON.parse(response.getObject());
                    Logger.info(['cordial.GetContacts--call', JSON.stringify(cordialContact)]);

                    var cID = cordialContact[0].attributes.cID;
                    Transaction.wrap(function () {
                      contact.custom.cordialCID = cID;
                    });

                    callCordialUpdateContacts(contact);
                  }, function () {
                    Logger.info(['create-new-Contacts--call', JSON.stringify(contact)]);
                    cordial.PostContacts(contact, function (response) {
                        Logger.info(['postContacts--ok', JSON.stringify(response)]);
                        if ('cID' in response) {
                            Transaction.wrap(function () {
                              contact.custom.cordialCID = response.cID;
                            });
                        }
                    }, function (response) {
                      errorFlag = true;
                      Logger.error('Import customer error err-on-post {0}', response);
                    });
                  });
            }

            counter++;
        }
      } //searchProfiles.hasNext()

      if (!errorFlag) {
        newCordialCustomersDateUpdate = helper.getTime();

        try {
            Transaction.wrap(function () {
                Site.getCurrent().setCustomPreferenceValue('cordialCustomersDateUpdate',
                    newCordialCustomersDateUpdate);
            });
            Transaction.wrap(function () {
                lastRunObject.custom.lastRunTime = newCordialCustomersDateUpdate;
            });
        } catch (e) {
            Logger.info(['finish-exportCustomers', newCordialCustomersDateUpdate]);
        }
      } else {
        throw new Error('Job failed due to error(s) above.');
      }
} //endfunction:exportCustomers


/**
 *
 * Test Run: export customers file
 **/
function exportCustomersFileTestRun(options) {
  var searchProfiles;
  var newCordialCustomersDateUpdate = helper.getTime();

  Logger.info(['call-exportCustomersFileTestRun', newCordialCustomersDateUpdate, 1000]);
  searchProfiles = CustomerMgr.searchProfiles('creationDate <= {0}', null, newCordialCustomersDateUpdate);

  exportCustomersFileRunner(searchProfiles, options, 1000);
}

/**
 *
 * This function export customers files
 **/
function exportCustomersFile(options) {
   var searchProfiles;

   var newCordialCustomersDateUpdate = helper.getTime();

   var lastRunObject = CustomObjectMgr.getCustomObject('CORDIAL_LASTRUN', 'cordialCustomersDateUpdate');
   if (!lastRunObject) {
        Transaction.wrap(function () {
            lastRunObject =
                CustomObjectMgr.createCustomObject('CORDIAL_LASTRUN', 'cordialCustomersDateUpdate');
        });
   }

   var cordialCustomersDateUpdate = lastRunObject.custom.lastRunTime
        ? lastRunObject.custom.lastRunTime
        : Site.getCurrent().getCustomPreferenceValue('cordialCustomersDateUpdate');

   Logger.info(['exportCustomersFile--run', cordialCustomersDateUpdate, newCordialCustomersDateUpdate]);

   if (!cordialCustomersDateUpdate) {
        cordialCustomersDateUpdate = newCordialCustomersDateUpdate;

        searchProfiles = CustomerMgr.searchProfiles(
            'creationDate < {0}',
            null,
            cordialCustomersDateUpdate
        );
   } else {
        searchProfiles = CustomerMgr.searchProfiles(
            'lastModified > {0}',
            null,
            cordialCustomersDateUpdate
        );
   }

    Transaction.wrap(function () {
        lastRunObject.custom.lastRunTime = newCordialCustomersDateUpdate;
    });
    Transaction.wrap(function () {
        Site.getCurrent().setCustomPreferenceValue('cordialCustomersDateUpdate',
            newCordialCustomersDateUpdate);
    });

    exportCustomersFileRunner(searchProfiles, options, 300000);
}

/**
 *  Generate Customer CSV for Export
 **/
function exportCustomersFileRunner(searchProfiles, options, limit) {
  var counter = 0;
  var newCordialCustomersDateUpdate = helper.getTime();
  var errorFlag = false;

  if (!limit) {
    limit = 9999999;
  }

  if (searchProfiles.hasNext()) {
        // Create CSV
        var siteID = Site.getCurrent().getID(),
            exportFolderPath = File.IMPEX + "/cordial/",
            exportFolder = new File(exportFolderPath);

        var exportFilename = exportFolder.fullPath + "customers_"
            + helper.getTimestamp() + ".csv";

        var exportFile = new File(exportFilename),
            fileWriter = new FileWriter(exportFile, "utf-8"),
            writer = new CSVStreamWriter(fileWriter, ',', '"'),
            isHeaders = false,
            customersProcessed = [],
            columnNames = [];
        Logger.info(['call-exportCustomersFile', exportFile]);

        while (searchProfiles.hasNext() && (counter < limit)) {
              var contact = searchProfiles.next();

              var customerMapping = new CustomerMapping();
              var json = Site.getCurrent().getCustomPreferenceValue('cordialCustomerMapping');
              var newJson = json.replace(/([a-zA-Z0-9]+?):/g, '"$1":');
              newJson = newJson.replace(/'/g, '"');
              var cordialCustomerMapping = JSON.parse(newJson);
              var cordialCustomer = customerMapping.execute(contact, cordialCustomerMapping);
              var flattenCordialCustomer = helper.flattenObject(cordialCustomer);

              // Write export progress to log
              if (counter % 2000 === 0 && counter) {
                    // Submit export CSV partially
                    // By small parts
                    // https://cordial.atlassian.net/browse/SF-62
                    writer.close();
                    fileWriter.close();

                    cordial.PostJobCustomers(
                          exportFile.fullPath, columnNames, options,
                          function () {
                            Logger.info('customersExport file is submitted: ' + exportFilename);
                          },
                          function (response) {
                            errorFlag = true;
                            Logger.error('Post job error {0}', response);
                          }
                    );

                    // Update customer date flag
                    customersProcessed.forEach(function(contact) {
                      //Transaction.wrap(function () {
                        //contact.custom.cordialLastExported = newCordialCustomersDateUpdate;
                      //});
                    });
                    customersProcessed = [];

                    isHeaders = false;
                    exportFilename = exportFolder.fullPath + "customers_"
                        + helper.getTimestamp() + "_" + counter + ".csv";
                    exportFile = new File(exportFilename),
                    fileWriter = new FileWriter(exportFile, "utf-8"),
                    writer = new CSVStreamWriter(fileWriter, ',', '"'),

                    Logger.info(['exportCustomersFile-progress', counter + ' out of ' + searchProfiles.getCount() + ' customers processed']);
              }

              // Write CSV header
              if (!isHeaders) {
                    //columnNames = Object.keys(flattenCordialCustomer);
                    columnNames = [];
                    cordialCustomerMapping.forEach(function (mSpecial) {
                      columnNames.push(mSpecial.cordial);
                    });

                    writer.writeNext(columnNames);
                    isHeaders = true;
              } //isHeaders

              // Write customer info
              var values = [];

              columnNames.forEach(function(name) {
                if (flattenCordialCustomer[name] === undefined) {
                  //values.push('undefined');
                  values.push('');
                } else {
                  if (typeof flattenCordialCustomer[name] === 'boolean') {
                    values.push(flattenCordialCustomer[name] ? 1 : 0);
                  } else if (name === 'birthday' || name === 'nextbirthday') {
                    values.push(flattenCordialCustomer[name].toISOString().split('T')[0]);
                  } else {
                    values.push(flattenCordialCustomer[name]);
                  }
                }
              });

              writer.writeNext(values);

              Transaction.wrap(function () {
                if (counter === 100) {
                    Logger.info(['exportCustomers-data', contact.custom.cordialLastExported,
                        contact.lastModified]);
                }
                customersProcessed.push(contact);
              });

              counter++;
        } //searchProfiles.hasNext

        // Clear resources
        writer.close();
        fileWriter.close();

        cordial.PostJobCustomers(
              exportFile.fullPath, columnNames, options,
              function () {
                Logger.info('customersExport file is submitted');
              },
              function (response) {
                errorFlag = true;
                Logger.error('Post job error {0}', response);
              }
        );

        Logger.info(['call-exportCustomersFile-finished', exportFilename, limit, counter]);
  }

  if (errorFlag) {
    throw new Error('Job failed due to error(s) above.');
  }
}


module.exports = {
  exportCustomers: exportCustomers,
  exportCustomersFile: exportCustomersFile,
  exportCustomersFileTestRun: exportCustomersFileTestRun
};
