'use strict';

// updated 2022-09-09
var ProductMgr = require('dw/catalog/ProductMgr');
var ProductInventory = require('dw/catalog/ProductInventoryMgr');

var Site = require('dw/system/Site');
var ProductMapping = require('int_cordial_core/cartridge/scripts/lib/mapping/product');
var Transaction = require('dw/system/Transaction');
var Logger = require('dw/system/Logger');
var CustomObjectMgr = require('dw/object/CustomObjectMgr');
var FileWriter = require('dw/io/FileWriter');
var File = require('dw/io/File');
var cordial = require('~/cartridge/scripts/services/cordial');
var helper = require('~/cartridge/scripts/lib/mapping/helper/helper');

/**
 * exportProducts
 * This function export products
 */
function exportProducts(options) {
	  var searchProducts;

	  var searchInventory = ProductInventory.getInventoryList();

      var lastRunObject = CustomObjectMgr.getCustomObject('CORDIAL_LASTRUN', 'cordialProductsDateUpdate');
      if (!lastRunObject) {
        Transaction.wrap(function () {
            lastRunObject =
                CustomObjectMgr.createCustomObject('CORDIAL_LASTRUN', 'cordialProductsDateUpdate');
        });
      }

      var cordialProductsDateUpdate = lastRunObject.custom.lastRunTime
            ? lastRunObject.custom.lastRunTime
            : Site.getCurrent().getCustomPreferenceValue('cordialProductsDateUpdate');

	  var errorFlag = false;

	  searchProducts = ProductMgr.queryAllSiteProductsSorted();
	  Logger.info(['Call-exportProducts', cordialProductsDateUpdate]);

	  while (searchProducts.hasNext()) {
		var product = searchProducts.next();

		if (cordialProductsDateUpdate) {
		  var productUpdatedRecently = product.lastModified >= cordialProductsDateUpdate;

          if (!productUpdatedRecently && product.onlineTo) {
              productUpdatedRecently = cordialProductsDateUpdate < product.onlineTo
                && product.onlineTo < helper.getTime();
          }

          if (!productUpdatedRecently && product.onlineFrom) {
              productUpdatedRecently = cordialProductsDateUpdate < product.onlineFrom
                && product.onlineFrom < helper.getTime();
          }

		  // Check inventory recent changes
		  var inventoryDetails = searchInventory.getRecord(product);

		  if (inventoryDetails)
		  if (inventoryDetails.getAllocationResetDate() >= cordialProductsDateUpdate) {
			Logger.info(['product-update-inventoryDetails', product, inventoryDetails]);
			productUpdatedRecently = true;
		  }

		  if (productUpdatedRecently) {
			Logger.info(['exportProducts-cordial.PostProducts', product, cordialProductsDateUpdate,
                product.custom.cordialLastExported, product.onlineFrom, product.onlineTo]);

			cordial.PostProducts(
			  product,
			  function () {},
			  function (response) {
				errorFlag = true;
				Logger.error('Import product error {0} on post', response);
			  }
			);

            Transaction.wrap(function(){
                product.custom.cordialLastExported = cordialProductsDateUpdate;
            });

			if (product.constructor.name === 'dw.catalog.Variant') {
			  cordial.PostProducts(
				product.getMasterProduct(),
				function () {},
				function (response) {
				  errorFlag = true;
				  Logger.error('Import product error {0} on post', response);
				}
			  );
			}
		  }
		} else {
        cordial.PostProducts(product, function () {}, function (response) {
          errorFlag = true;
          Logger.error('Import product error {0} on post', response);
        });
      }
	  }

	  var newCordialProductsDateUpdate = helper.getTime();

	  if (!errorFlag) {
        try {
                Transaction.wrap(function () {
                    Site.getCurrent().setCustomPreferenceValue('cordialProductsDateUpdate',
                        newCordialProductsDateUpdate);
                });
                Transaction.wrap(function () {
                    lastRunObject.custom.lastRunTime = newCordialProductsDateUpdate;
                });
        } catch (e) {
                Logger.info(['finish-exportProducts', newCordialProductsDateUpdate]);
        }
    } else {
      throw new Error('Job failed due to error(s) above.');
    }
}

/**
 *
 * This function export products files
 */
function exportProductsFile(options) {
  var searchProducts;
  var newCordialProductsDateUpdate = helper.getTime();

  var lastRunObject = CustomObjectMgr.getCustomObject('CORDIAL_LASTRUN', 'cordialProductsDateUpdate');
  if (!lastRunObject) {
    Transaction.wrap(function () {
        lastRunObject =
            CustomObjectMgr.createCustomObject('CORDIAL_LASTRUN', 'cordialProductsDateUpdate');
    });
  }

  var cordialProductsDateUpdate = lastRunObject.custom.lastRunTime
        ? lastRunObject.custom.lastRunTime
        : Site.getCurrent().getCustomPreferenceValue('cordialProductsDateUpdate');

  if (!cordialProductsDateUpdate) {
            cordialProductsDateUpdate = newCordialProductsDateUpdate;
            try {
                Transaction.wrap(function () {
                    Site.getCurrent().setCustomPreferenceValue('cordialProductsDateUpdate',
                        newCordialProductsDateUpdate);
                });
                Transaction.wrap(function () {
                    lastRunObject.custom.lastRunTime = newCordialProductsDateUpdate;
                });
            } catch (e) {
                Logger.info(['finish-exportProducts', newCordialProductsDateUpdate]);
            }
  }

  searchProducts = ProductMgr.queryAllSiteProductsSorted();

  if (searchProducts.hasNext()) {
		// Create CSV for export
		var siteID = Site.getCurrent().getID();
		var exportFolderPath = File.IMPEX + "/cordial/";
		var exportFolder = new File(exportFolderPath);
    var errorFlag = false;

		if (!exportFolder.exists()) {
		  exportFolder.mkdirs();
		}

		var exportFilename = exportFolder.fullPath + File.SEPARATOR
            + "products_" + helper.getTimestamp() + ".json";
		var exportFile = new File(exportFilename);
		var fileWriter = new FileWriter(exportFile, "utf-8");
        var productsProcessed = [];
        var counter = 0;
		Logger.info(['Call-exportProductsFile', searchProducts.count, exportFilename]);

		while (searchProducts.hasNext()) {
		  var product = searchProducts.next();
          var cordialLastExported;

          try {
            cordialLastExported = product.custom.cordialLastExported;
          } catch (e) {
            cordialLastExported = '';
            product.custom.put('cordialLastExported', '');
          }

          var productUpdatedRecently = product.lastModified >= cordialProductsDateUpdate;

          if (!productUpdatedRecently && product.onlineTo) {
              productUpdatedRecently = cordialProductsDateUpdate < product.onlineTo
                && product.onlineTo < helper.getTime();
          }

          if (!productUpdatedRecently && product.onlineFrom) {
              productUpdatedRecently = cordialProductsDateUpdate < product.onlineFrom
                && product.onlineFrom < helper.getTime();
          }

          if (!cordialLastExported || productUpdatedRecently) {
              counter ++;

              var productMapping = new ProductMapping();
              var cordialProduct = productMapping.execute(product)

              if (counter % 400 === 0) {
                    Logger.info(['exportProducts-details',
                        cordialLastExported, cordialProductsDateUpdate, product.lastModified,
                        productUpdatedRecently]);
              }

              if (counter % 2000 === 0) {
                    fileWriter.close();

                    cordial.PostJobProducts(
                        exportFile.fullPath,
                        options,
                        function () {},
                        function (response) {
                          errorFlag = true;
                          Logger.error('Post job error {0}', response);
                        }
                    );

                    productsProcessed.forEach(function(product){
                        Transaction.wrap(function(){
                            product.custom.cordialLastExported = cordialProductsDateUpdate;
                        });
                    });
                    productsProcessed = [];

                    exportFilename = exportFolder.fullPath + File.SEPARATOR
                            + "products_" + helper.getTimestamp()
                            + "_" + counter
                            + ".json";
                    exportFile = new File(exportFilename);
                    fileWriter = new FileWriter(exportFile, "utf-8");

                    Logger.info(['export-products-progress', counter + ' out of '
                                  + searchProducts.getCount() + ' products processed']);
              }

              fileWriter.writeLine(JSON.stringify(cordialProduct));
              productsProcessed.push(product);
          }
		}
		fileWriter.close();

		cordial.PostJobProducts(exportFile.fullPath, options, function () {}, function (response) {
      errorFlag = true;
      Logger.error('PostJobProducts action error {0}', response);
		});

        productsProcessed.forEach(function(product){
            Transaction.wrap(function(){
                product.custom.cordialLastExported = cordialProductsDateUpdate;
            });
        });
        productsProcessed = [];
  }

  newCordialProductsDateUpdate = helper.getTime();

  try {
        Transaction.wrap(function () {
            Site.getCurrent().setCustomPreferenceValue('cordialProductsDateUpdate',
                newCordialProductsDateUpdate);
        });
        Transaction.wrap(function () {
            lastRunObject.custom.lastRunTime = newCordialProductsDateUpdate;
        });
  } catch (e) {
        Logger.info(['finish-exportProducts', newCordialProductsDateUpdate]);
  }

  if (errorFlag) {
    throw new Error('Job failed due to error(s) above.');
  }
}

module.exports = {
  exportProducts: exportProducts,
  exportProductsFile: exportProductsFile
};
