/**
 * This controller provides different endpoints for calling Cordial services
 *
 * @module  controllers/cordial
 */

'use strict';

/* Script Modules */
var cordial = require('int_cordial_core/cartridge/scripts/services/libCordial');
var CustomerMapping = require('int_cordial_core/cartridge/scripts/lib/mapping/customer');
var OrderMapping = require('int_cordial_core/cartridge/scripts/lib/mapping/order');
var ProductMapping = require('int_cordial_core/cartridge/scripts/lib/mapping/product');
var CartMapping = require('int_cordial_core/cartridge/scripts/lib/mapping/cart');
var Site = require('dw/system/Site');
var Logger = require('dw/system/Logger');

function returnResponse(response, success, failed) {
    if (response.status === "OK") {
        // all good
        success(response);
    } else {
        failed(response);
    }
}

function getContacts(params, success, failed) {
    var service = cordial.getContacts();

    var result = service.call(params);

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - get Contacts');
        logger.debug('Calling params ' + params);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function postContacts(contact, success, failed) {
    var Transaction = require('dw/system/Transaction');
    var cordialCustomersAddToEmailList = dw.system.Site.getCurrent().getCustomPreferenceValue('cordialCustomersAddToEmailList');

    var cordialCustomerSubscribeStatusDef = dw.system.Site.getCurrent().getCustomPreferenceValue('cordialCustomerSubscribeStatusDef');
    var logger = Logger.getLogger('Cordial', 'cordial - get Contacts');

    try {
        if (!contact.custom['cordialSubscribed']) {
            Transaction.wrap(function () {
                contact.custom['cordialSubscribed'] = cordialCustomerSubscribeStatusDef;
            });
        }
    } catch (e) {
        logger.error('cordialCustomersAddToEmailLst--failed:' + e.message);
    }

    var service = cordial.postContacts();
    var customerMapping = new CustomerMapping();
    var json = Site.getCurrent().getCustomPreferenceValue('cordialCustomerMapping');
    var newJson = json.replace(/([a-zA-Z0-9]+?):/g, '"$1":');
    newJson = newJson.replace(/'/g, '"');
    var cordialCustomerMapping = JSON.parse(newJson);
    var cordialCustomer = customerMapping.execute(contact, cordialCustomerMapping);

    var result = service.call(JSON.stringify(cordialCustomer));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post Contacts');
        logger.debug('Calling params--dump ' + JSON.stringify(cordialCustomer));
        //logger.debug('Calling params--dump ' + JSON.stringify(cordialCustomerMapping));
        //logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function updateContacts(contact, success, failed) {
    var Transaction = require('dw/system/Transaction');
    var cordialCustomersAddToEmailList = dw.system.Site.getCurrent().getCustomPreferenceValue('cordialCustomersAddToEmailList');
    var cordialCustomerSubscribeStatusDef =
        dw.system.Site.getCurrent().getCustomPreferenceValue('cordialCustomerSubscribeStatusDef');

    if (cordialCustomersAddToEmailList) {
        if (!contact.custom.cordialSubscribed) {
            Transaction.wrap(function () {
                contact.custom.cordialSubscribed = cordialCustomerSubscribeStatusDef;
            });
        }
    }

    var service = cordial.updateContacts();
    var customerMapping = new CustomerMapping();
    var json = Site.getCurrent().getCustomPreferenceValue('cordialCustomerMapping');
    var newJson = json.replace(/([a-zA-Z0-9]+?):/g, '"$1":');
    newJson = newJson.replace(/'/g, '"');
    var cordialCustomerMapping = JSON.parse(newJson);

    var cordialCustomer = customerMapping.execute(contact, cordialCustomerMapping);

    var logger;
    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        logger = Logger.getLogger('Cordial', 'cordial - update Contacts');
        logger.debug('Calling params ' + JSON.stringify(cordialCustomer));
    }

    var result = service.call(JSON.stringify(cordialCustomer), contact.custom.cordialCID);

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function getOrders(params, success, failed) {
    var service = cordial.getOrders();

    let reqObj = params || new Object();;

    var result = service.call(reqObj);

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - get Orders');
        logger.debug('Calling params ' + reqObj);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function postOrders(order, success, failed, oldOrder) {
    var orderMapping = new OrderMapping();
    var cordialOrder = orderMapping.execute(order);
    var result;

    if (oldOrder) {
        var service = cordial.updateOrder();
        result = service.call(JSON.stringify(cordialOrder), cordialOrder.orderID);
        Logger.info('postOrders--call {0} {1}', oldOrder.status, cordialOrder.status);

        if (oldOrder.status !== cordialOrder.status) {
            postContactactivities(cordialOrder, cordialOrder.status, order.getLastModified())
        }
        if (oldOrder.properties.shippingStatus !== cordialOrder.properties.shippingStatus) {
            postContactactivities(cordialOrder, cordialOrder.properties.shippingStatus, order.getLastModified())
        }
    } else {
        var service = cordial.postOrder();
        result = service.call(JSON.stringify(cordialOrder));
        postContactactivities(cordialOrder, cordialOrder.status, order.getLastModified());
        postContactactivities(cordialOrder, cordialOrder.properties.shippingStatus, order.getLastModified())
    }

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post Orders');
        logger.debug('Calling params ' + cordialOrder);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function getProducts(params, success, failed) {
    var service = cordial.getProducts();

    let reqObj = params || new Object();;

    var result = service.call(JSON.stringify(reqObj));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - get Products');
        logger.debug('Calling params ' + reqObj);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

/**
 *  contact activity api
 *  with event name = order
 **/
function postContactactivities(order, status, lastModified) {
    var service = cordial.postContactActivities();

    var contactActivities = {
        'a': 'order',
        'email': order.email,
        'ats':  lastModified,
        'properties': {
            'orderID': order.orderID,
            'status': status
        }
    };
    Logger.info('postContactactivities--call {0}', JSON.stringify(contactActivities));

    var result = service.call(JSON.stringify(contactActivities));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post Contact activities');
        logger.debug('Calling params ' + contactActivities);
        logger.debug('Service result ' + result);
    }
}


function postProducts(product, success, failed) {
    Logger.info(['serviceCordial-postProducts', product]);
    var service = cordial.postProducts();
    var productMapping = new ProductMapping();
    var cordialProduct = productMapping.execute(product);

    var result = service.call(JSON.stringify(cordialProduct));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post Products');
        logger.debug('Calling params ' + cordialProduct);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function updateProducts(product, success, failed) {
    var service = cordial.updateProducts();
    var productMapping = new ProductMapping();
    var cordialProduct = productMapping.execute(product);

    var result = service.call(JSON.stringify(cordialProduct));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - update Products');
        logger.debug('Calling params ' + cordialProduct);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function updateCart(cart, customer, success, failed) {
    var serviceUpdateCart = cordial.updateCart();
    var serviceDeleteCart = cordial.deleteCart();
    var cartMapping = new CartMapping();
    var cordialCart = cartMapping.execute(cart);
    var cid;
    if (customer.custom.cordialCID) {
        cid  = customer.custom.cordialCID;
    } else {
        getContacts({"email": customer.getProfile().getEmail()}, function(data) {
            cid  = data.attributes.cID;
        }, failed);

    }

    serviceDeleteCart.call(JSON.stringify(cordialCart), cid);
    var result = serviceUpdateCart.call(JSON.stringify(cordialCart), cid);

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - update Cart');
        logger.debug('Calling params ' + cordialCart, cid);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function postJobCustomers(url, columnNames, options, success, failed) {
    var service = cordial.postJobCustomers();

    var params = {
            "importName": "ImportContactsSFCC",
            "source": {
                "transport": "https",
                "username": options.username,
                "password": options.password,
                "url": 'https://' + Site.getCurrent().getHttpsHostName() + '/on/demandware.servlet/webdav/Sites' + url
            },
            "hasHeader": true,
            "columns": columnNames,
            "confirmEmail": options.confirmEmail,
            "nullMarker": "ignore"
    }

    var result = service.call(JSON.stringify(params));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post job customers');
        logger.debug('Calling params ' + params);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function postJobProducts(url, options, success, failed) {
    var service = cordial.postJobProducts();

    var params = {
            "importName": "ImportProductsSFCC",
            "source": {
                "transport": "https",
                "username": options.username,
                "password": options.password,
                "url": 'https://' + Site.getCurrent().getHttpsHostName()
                    + '/on/demandware.servlet/webdav/Sites' + url
            },
            "hasHeader": true,
            "columns": [],
            "confirmEmail": options.confirmEmail
    }

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post job products');
        logger.debug('Calling params ' + params);
    }

    var result = service.call(JSON.stringify(params));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post job products');
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function postJobOrders(url, options, success, failed) {
    var service = cordial.postJobOrders();

    var params = {
            "importName": "ImportOrdersSFCC",
            "source": {
                "transport": "https",
                "username": options.username,
                "password": options.password,
                "url": 'https://' + Site.getCurrent().getHttpsHostName() + '/on/demandware.servlet/webdav/Sites' + url
            },
            "hasHeader": false,
            "columns": [],
            "confirmEmail": options.confirmEmail
    }

    var result = service.call(JSON.stringify(params));

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - post job orders');
        logger.debug('Calling params ' + params);
        logger.debug('Service result ' + result);
    }

    return returnResponse(result, success, failed);
}

function postProductList(productList, supplementsName, success, failed) {
    var serviceGetProductList = cordial.getProductList();
    var resultGetProductList = serviceGetProductList.call(supplementsName);

    if (resultGetProductList.status === 'ERROR') {
        var keys = {
            "key": supplementsName,
            "name": supplementsName,
            "contactObject": true,
            "indexes": [
              {
                "field": "productID",
                "type": "array"
              },
              {
                "field": "qnt",
                "type": "array"
              },
              {
                "field": "price",
                "type": "array"
              },
              {
                "field": "category",
                "type": "array"
              },
              {
                "field": "sfccCreationDate",
                "type": "date"
              },
              {
                "field": "sfccLastUpdateDate",
                "type": "date"
              }
            ]
          }
        var servicePostProductList = cordial.postProductList();
        var resultPostProductList = servicePostProductList.call(JSON.stringify(keys));
        if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
            var logger = Logger.getLogger('Cordial', 'cordial - post Product List');
            logger.debug('Service result ' + resultPostProductList);
        }
    }

    var productItems = productList.getProductItems();

    var service = cordial.deleteProductListRecord();
    var result = service.call(productList.getID(), supplementsName);

    if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
        var logger = Logger.getLogger('Cordial', 'cordial - delete Product List record');
        logger.debug('Calling params ' + productList.getID() + supplementsName);
        logger.debug('Service result ' + result);
    }

    if (productItems.length > 0) {
        var productIDArray = [];
        var qntArray = [];
        var priceArray = [];
        var categoryArray = [];
        productItems.toArray().forEach(function(productItem){
            productIDArray.push(productItem.productID);
            qntArray.push(productItem.quantity.value);
            priceArray.push(productItem.product.getPriceModel().getPrice().getValue());
            var catergoryArray = [];
            if (productItem.product.categorized) {
                var categories = productItem.product.getCategories().toArray();
                categories.forEach(function(category){
                    catergoryArray.push(category.getID())
                })
                if (catergoryArray.length === 0 && productItem.product.getClassificationCategory()) {
                    catergoryArray.push(productItem.product.getClassificationCategory().getID())
                }
                if (catergoryArray.length === 0 && productItem.product.getPrimaryCategory()) {
                    catergoryArray.push(productItem.product.getPrimaryCategory().getID())
                }
            } else {
                catergoryArray.push('not categorized')
            }
            catergoryArray.join(',');
            categoryArray.push(catergoryArray);
        })
        var productListMap = {
            id: productList.getID(),
            email: productList.getOwner().getProfile().getEmail(),
            productID: productIDArray,
            qnt: qntArray,
            price: priceArray,
            category: categoryArray,
            sfccCreationDate: productList.getCreationDate(),
            sfccLastUpdateDate: productList.getLastModified()
        };
        var service = cordial.postProductListRecord();
        var result = service.call(JSON.stringify(productListMap), supplementsName);

        if (dw.system.Site.getCurrent().getCustomPreferenceValue('cordialEnabledLogs')) {
            var logger = Logger.getLogger('Cordial', 'cordial - post Product List record');
            logger.debug('Calling params ' + productListMap);
            logger.debug('Service result ' + result);
        }
    }
}

/**
 * @see {@link module:controllers/Cordial~getContacts} */
 exports.GetContacts = getContacts;
 /**
 * @see {@link module:controllers/Cordial~postContacts} */
 exports.PostContacts = postContacts;
 /**
 /**
 * @see {@link module:controllers/Cordial~updateContacts} */
 exports.UpdateContacts = updateContacts;
 /**
 * @see {@link module:controllers/Cordial~getOrders} */
 exports.GetOrders = getOrders;
 /**
 * @see {@link module:controllers/Cordial~postOrders} */
 exports.PostOrders = postOrders;
 /**
 * @see {@link module:controllers/Cordial~getProducts} */
 exports.GetProducts = getProducts;
 /**
 * @see {@link module:controllers/Cordial~postProducts} */
 exports.PostProducts = postProducts;
  /**
 /**
 * @see {@link module:controllers/Cordial~updateProducts} */
 exports.UpdateProducts = updateProducts;
 /**
 * @see {@link module:controllers/Cordial~updateCart} */
 exports.UpdateCart = updateCart;
  /**
 * @see {@link module:controllers/Cordial~PostJobCustomers} */
 exports.PostJobCustomers = postJobCustomers;
   /**
 * @see {@link module:controllers/Cordial~postJobProducts} */
 exports.PostJobProducts = postJobProducts;
    /**
 * @see {@link module:controllers/Cordial~postJobOrders} */
 exports.PostJobOrders = postJobOrders;
     /**
 * @see {@link module:controllers/Cordial~PostProductList} */
 exports.PostProductList = postProductList;
