'use strict';

var Logger = require('dw/system/Logger');
var productMgr = require('dw/catalog/ProductMgr');
var orderMgr = require('dw/order/OrderMgr');
var basketMgr = require('dw/order/BasketMgr');
var URLUtils = require('dw/web/URLUtils');

/**
 * Prepares Viewed Product Object and set necessary details
 *
 * @param viewedProduct
 * @returns {Object}
 **/
function prepareViewedProductEventData(viewedProduct) {
  var logger = Logger.getLogger('Cordial', 'cordialUtils - prepareViewedProductEventData()');
  var crdlData = {};
  try {
    crdlData.event = 'product_view';
    var ProductMapping = require('../../../scripts/lib/mapping/product');
    var productMapping = new ProductMapping();
    var cordialProduct = productMapping.execute(viewedProduct);
    crdlData.data = cordialProduct;

    return crdlData;
  } catch (e) {
    crdlData.data = {};
    crdlData.data.debug_error = [e.message, e.lineNumber];
    logger.error('prepareViewedProductEventData() failed for product: ' + viewedProduct.getID() + '. Error: ' + e.message);
  }
}

/**
 * Creating data layer from the basket for checkout start event
 * @param basket
 * @returns datalayer
 */
function prepareCheckoutEventForCordial(currentBasket) {
  var logger = Logger.getLogger('Cordial', 'cordialUtils - prepareCheckoutEventForCordial()');
  var crdlData = {};
  try {
    crdlData.event = 'Started Checkout';
    var CartMapping = require('../../../scripts/lib/mapping/cart');
    var cartMapping = new CartMapping();
    var cordialCart = cartMapping.execute(currentBasket);
    crdlData.data = cordialCart;
    return crdlData;
  } catch (e) {
    crdlData.data = {};
    crdlData.data.debug_error = [e.message, e.lineNumber];
    logger.error('prepareCheckoutEventForCordial() failed for basket. Error: ' + e.message);
  }
}

/**
 * Creating data layer from the order object and send to cordial
 * @param order
 * @returns
 */
function prepareOrderConfirmationEventForCordial(currentOrder) {
  var logger = Logger.getLogger('Cordial', 'cordialUtils - prepareOrderConfirmationEventForCordial()');
  var crdlData = {};
  try {
    crdlData.event = 'new_order';
    var OrderMapping = require('../../../scripts/lib/mapping/order');
    var orderMapping = new OrderMapping();
    var cordialOrder = orderMapping.execute(currentOrder);
    crdlData.data = cordialOrder;

    return crdlData;
  } catch (e) {
    crdlData.data = {};
    crdlData.data.debug_error = [e.message, e.lineNumber];
    logger.error('prepareOrderConfirmationEventForCordial() failed for order: ' + currentOrder.getOrderNo() + '. Error: ' + e.message);
  }
}

/**
 * Creating data layer from the basket for add to cart event.
 * @param basket
 * @returns datalayer
 */
function prepareAddToCartEventForCordial(currentBasket, basketHasLength) {
  var logger = Logger.getLogger('Cordial', 'cordialUtils - prepareAddToCartEventForCordial()');
  var crdlData = {};

  try {
    crdlData.event = 'updating_cart';
    var CartMapping = require('../../../scripts/lib/mapping/cart');
    var cartMapping = new CartMapping();
    var cordialCart = cartMapping.execute(currentBasket);
    crdlData.data = cordialCart;

    return crdlData;
  } catch (e) {
    crdlData.data = {};
    crdlData.data.debug_error = [e.message, e.lineNumber];
    logger.error('prepareAddToCartEventForCordial() failed for basket. Error: ' + e.message);
  }
}


/**
 * Prepare data's needs to be send to cordial in crdlData object
 */
var buildDataLayer = function () {
  var crdlData = {};
  var
    basketHasLength,
    currentOrder,
    viewedProduct;

  crdlData.data = {};
  crdlData.data.debug_error = '';

  var httpParameterMap = request.httpParameterMap;
  var pageContext = httpParameterMap.pagecontexttype;
  var isCheckout = request.httpReferer.match(/(Checkout-Begin|placeOrder|COBilling-Billing)/)
    ? true : false;
  var pageProductID = httpParameterMap.productid;
  var orderID = httpParameterMap.orderno;
  var searchResultsCount = httpParameterMap.searchresultscount;
  var searchTerm = httpParameterMap.searchterm.value;
  var pageCategoryId = httpParameterMap.pagecgid.value;
  var logger = Logger.getLogger('Cordial', 'cordialUtils - buildDataLayer()');
  //logger.debug('cordialUtils--pagecontexttype21:' + isCheckout);
  //logger.debug('cordialUtils--pagecontexttype3:' + request.httpQueryString + ':' + request.httpReferer);

  try {
    // Order Placed Event
    // 2022-08-23
    // pageContext = do not have info about orderconfirmation page
    if (isCheckout && orderID) {
      currentOrder = orderMgr.getOrder(orderID);
      crdlData = prepareOrderConfirmationEventForCordial(currentOrder);

      return crdlData
    }

    crdlData.controllerName = pageContext.toString();

    // Viewed Product event
    if (pageContext == 'product' || pageProductID) {
      viewedProduct = productMgr.getProduct(pageProductID);
      if (viewedProduct) {
        crdlData = prepareViewedProductEventData(viewedProduct);
        return crdlData
      }
    }

    // Category Viewed event
    if (pageContext == 'search' && !empty(pageCategoryId)) {
      crdlData.event = 'category_view';
      crdlData.data = {
        'url': URLUtils.https('Search-Show', 'cgid', pageCategoryId).toString()
      }
      return crdlData
    }

    // Search page
    if (!empty(searchTerm)) {
      crdlData.event = 'search_page';
      crdlData.data = {
        'url': URLUtils.https('Search-Show', 'q', pageCategoryId).toString()
      }
      return crdlData
    }
  } catch (e) {
    crdlData.data.debug_error = [e.message, e.lineNumber];
    logger.error('buildDataLayer() failed for build data layer. Error: ' + e.message);
  }

  return crdlData;
};

/**
 * Prepare data's needs to be send to cordial in crdlData object for add to cart
 */
var buildCartDataLayer = function () {
  var crdlData = {};
  var isValidBasket,
    basketHasLength;
  var logger = Logger.getLogger('Cordial', 'cordialUtils - buildCartDataLayer()');
  try {
    isValidBasket = (basketMgr.getCurrentBasket());

    if (isValidBasket) {
      basketHasLength = (basketMgr.getCurrentBasket().getProductLineItems().toArray().length >= 1);
    }

    if (isValidBasket) {
      crdlData = prepareAddToCartEventForCordial(basketMgr.getCurrentBasket(), basketHasLength);
    }

    return crdlData;
  } catch (e) {
    crdlData.data = {};
    crdlData.data.debug_error = [e.message, e.lineNumber];
    logger.error('buildCartDataLayer() failed for build cart data layer. Error: ' + e.message);

  }
};

module.exports = {
  prepareViewedProductEventData: prepareViewedProductEventData,
  prepareCheckoutEventForCordial: prepareCheckoutEventForCordial,
  prepareOrderConfirmationEventForCordial: prepareOrderConfirmationEventForCordial,
  prepareAddToCartEventForCordial: prepareAddToCartEventForCordial,
  buildDataLayer: buildDataLayer,
  buildCartDataLayer: buildCartDataLayer,
};
